module net.BurtonRadons.parse.statement;

/** The base statement object.  Statements hold declarations, conditionals, and miscellany. */
class Statement
{
    import net.BurtonRadons.parse.declaration;
    import net.BurtonRadons.parse.lexer;
    import net.BurtonRadons.parse.scope;
    
    Statement parent;
    Marker mark;
    Scope scope;

    /** The first semantic pass registers type names in the parent scope. */
    Statement semantic0 (Scope scope)
    {
        return this;
    }

    /** The second semantic pass determines the solid types of everything. */
    Statement semantic1 (Scope scope)
    {
        return this;
    }

    /** The third semantic pass sets up the solid form of code, such as struct offsets, and tests code validity. */
    Statement semantic2 (Scope scope)
    {
        return this;
    }

    char [] toString ()
    {
        throw new Error (this.classinfo.name ~ ".toString () is unimplemented.");
        return null;
    }

    /** Return the function this statement is within or null if it is not in any. */
    FunctionDeclaration findFunction ()
    {
        Statement current = this;

        while (current !== null && (cast (FunctionDeclaration) current) === null)
            current = current.parent;
        if (current === null)
            return null;
        return cast (FunctionDeclaration) current;
    }
}

/** An expression statement. */
class ExpStatement : Statement
{
    import net.BurtonRadons.parse.expression;
    
    Expression exp; /**< Root of the expression. */

    /** Assign parameters. */
    this (Expression exp)
    {
        this.exp = exp;
    }

    override Statement semantic0 (Scope parent) { exp = exp.semantic0 (parent); return this; }
    override Statement semantic1 (Scope parent) { exp = exp.semantic1 (parent); return this; }
    override Statement semantic2 (Scope parent) { exp = exp.semantic2 (parent); return this; }
    override char [] toString () { return exp.toString () ~ ";"; }
}

/** A set of statements. */
class BlockStatement : Statement
{
    import net.BurtonRadons.parse.parser;
    
    /** Create the scope. */
    this (Parser parser)
    {
        this.scope = new Scope (parser);
    }

    override Statement semantic0 (Scope parent)
    {
        scope.parent = parent;
        scope.semantic0 ();
        return this;
    }

    override Statement semantic1 (Scope parent)
    {
        scope.parent = parent;
        scope.semantic1 ();
        return this;
    }

    override Statement semantic2 (Scope parent)
    {
        scope.parent = parent;
        scope.semantic2 ();
        return this;
    }
}

/** Fold test into a boolean; expand onTrue if it is true, expand onFalse if it is false. */
class IfElseStatement : Statement
{
    import net.BurtonRadons.parse.expression;
    
    Expression test; /**< Test expression. */
    Statement onTrue; /**< Statement executed when test is true. */
    Statement onFalse; /**< Statement executed when test is false, null if there is none. */

    /** Assign parameters. */
    this (Expression test, Statement onTrue, Statement onFalse)
    {
        this.test = test;
        this.onTrue = onTrue;
        this.onFalse = onFalse;
    }
}

/** Return the expression "value" from the function. */
class ReturnStatement : Statement
{
    import net.BurtonRadons.parse.expression;
    
    Expression value; /**< The value to return or null. */

    /** Assign parameters. */
    this (Expression value)
    {
        this.value = value;
    }

    Statement semantic0 (Scope scope)
    {
        if (value !== null)
            value = value.semantic0 (scope); 
        return this;
    }

    Statement semantic1 (Scope scope)
    {
        FunctionDeclaration decl = findFunction ();

        if (value !== null)
            value = value.semantic1 (scope);
        if (decl !== null && value !== null)
            value = value.implicitCastTo (scope, decl.type.next);
        return this;
    }
    
    Statement semantic2 (Scope scope)
    {
        if (value !== null)
            value = value.semantic2 (scope);
        return this;
    }
}

/** A loop statement; so long as "test" is true, execute "content". */
class WhileStatement : Statement
{
    import net.BurtonRadons.parse.expression;
    
    Expression test; /**< The expression to test. */
    Statement content; /**< Statement to execute while test is true. */

    /** Assign parameters. */
    this (Expression test, Statement content)
    {
        this.test = test;
        this.content = content;
    }

    Statement semantic0 (Scope scope)
    {
        test = test.semantic0 (scope);
        content = content.semantic0 (scope);
        return this;
    }

    Statement semantic1 (Scope scope)
    {
        test = test.semantic1 (scope);
        content = content.semantic1 (scope);
        return this;
    }

    Statement semantic2 (Scope scope)
    {
        test = test.semantic2 (scope);
        content = content.semantic2 (scope);
        return this;
    }
}

/** Search through the list for a case matching value; when found, start
  * execution at that point.  If the value is not found, execution starts at
  * default.  This uses fallthrough semantics; a case does not imply
  * a break before it.
  */

class SwitchStatement : Statement
{
    import net.BurtonRadons.parse.expression;
    
    Expression value; /**< The value to switch upon. */
    Statement content; /**< The switch body searched for cases. */

    /** Assign the parameters. */
    this (Expression value, Statement content)
    {
        this.value = value;
        this.content = content;
    }
}

/** A case entry for a switch statement.  If the value from switch matches the
  * value in case, it starts execution at this point.  A case does not have a 
  * break before it - code will execute through it.
  */

class CaseStatement : Statement
{
    import net.BurtonRadons.parse.expression;
    
    Expression value; /**< The value of the case. */

    /** Assign the parameters. */
    this (Marker mark, Expression value)
    {
        this.mark = mark;
        this.value = value;
    }
}

/** The default entry for a switch statement.  If the switch value has no equal
  * case, it starts execution at this point if it exists.  Otherwise it spawns a
  * runtime error.
  */

class DefaultStatement : Statement
{
    /** Assign the parameter. */
    this (Marker mark)
    {
        this.mark = mark;
    }
}

/** Throw an exception. */
class ThrowStatement : Statement
{
    import net.BurtonRadons.parse.expression;
    
    Expression value; /**< The value to throw. */

    /** Assign the parameters. */
    this (Marker mark, Expression value)
    {
        this.mark = mark;
        this.value = value;
    }
}

/** A label, appropriate for goto.  If a do/while, while, or switch immediately follows it, it can be used as a break or continue label. */
class LabelStatement : Statement
{
    Symbol name; /**< The symbol this defines. */

    /** Assign the parameters. */
    this (Marker mark, Symbol name)
    {
        this.mark = mark;
        this.name = name;
    }
}

/** A three-part loop statement which first calls an initialiser, runs the
  * test, runs the content if the test succeeds, and then runs the 
  * increment before looping back to the test.
  */

class ForStatement : Statement
{
    import net.BurtonRadons.parse.expression;
    
    Statement init; /**< Initialiser or null. */
    Expression test; /**< Expression tested to see whether we should run the content or null for always run. */
    Expression incr; /**< Increment run at the end of the content. */
    Statement content; /**< The body of code to run whenever test returns true or when test is null. */

    /** Assign the parameters. */
    this (Marker mark, Statement init, Expression test, Expression incr, Statement content)
    {
        this.mark = mark;
        this.init = init;
        this.test = test;
        this.incr = incr;
        this.content = content;
    }
}

/** Search the object for fields and methods first, as if executing a method within it. */
class WithStatement : Statement
{
    import net.BurtonRadons.parse.expression;
    
    Expression object; /**< The object to execute the statements on. */
    Statement content; /**< The body of code to run. */

    /** Assign the parameters. */
    this (Marker mark, Expression object, Statement content)
    {
        this.mark = mark;
        this.object = object;
        this.content = content;
    }
}

/** Write all data before and after the statement. */
class VolatileStatement : Statement
{
    Statement content; /**< The statement contained by the volatile. */

    /** Assign the parameters. */
    this (Marker mark, Statement content)
    {
        this.mark = mark;
        this.content = content;
    }
}